<?php
namespace App\Helpers;

class Pdu
{
    const ENCODING_GSM_7BIT = 7;
    const ENCODING_8BIT = 8;
    const ENCODING_UCS2 = 16;

    protected static $sevenbitdefault = array(
        '@', '£', '$', '¥', 'è', 'é', 'ù', 'ì', 'ò', 'Ç', "\n", 'Ø', 'ø', "\r", 'Å', 'å',
        'Δ', '_', 'Φ', 'Γ', 'Λ', 'Ω', 'Π', 'Ψ',
        'Σ', 'Θ', 'Ξ', '☝', 'Æ', 'æ', 'ß', 'É',
        ' ', '!', '"', '#', '¤', '%', '&', '\'', '(', ')', '*', '+', ',', '-', '.', '/',
        '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?',
        '¡', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O',
        'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', 'Ä', 'Ö', 'Ñ', 'Ü', '§',
        '¿', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o',
        'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', 'ä', 'ö', 'ñ', 'ü', 'à'
    );

    protected static $sevenbitextended = array(
        '\f' => 0x0A,   // <FF>
        '^'  => 0x14,   // CIRCUMFLEX ACCENT
        '{'  => 0x28,   // LEFT CURLY BRACKET
        '}'  => 0x29,   // RIGHT CURLY BRACKET
        '\\' => 0x2F,   // REVERSE SOLIDUS
        '['  => 0x3C,   // LEFT SQUARE BRACKET
        '~'  => 0x3D,   // TILDE
        ']'  => 0x3E,   // RIGHT SQUARE BRACKET
        '|'  => 0x40,   // VERTICAL LINE
        '€'  => 0x65    // EURO SIGN
    );

    /**
     * Parses and decodes full PDU string and returns array with sender, timestamp, encoding and message
     *
     * @param string $fullPdu
     * @return array
     */
    public function parseAndDecodeFullPdu(string $fullPdu): array
    {
        // Step 1: SMSC info length (in octets)
        $smcLenOctets = hexdec(substr($fullPdu, 0, 2));

        // Step 2: SMSC info (skip this many octets)
        $smcInfoLenHex = $smcLenOctets * 2;
        $index = 2 + $smcInfoLenHex;

        // Step 3: First octet of SMS-DELIVER (skip 1 octet)
        $index += 2;

        // Step 4: Sender address length in semi-octets (digits)
        $senderLenSemiOctets = hexdec(substr($fullPdu, $index, 2));
        $index += 2;

        // Step 5: Sender type of address (1 octet)
        $senderTypeOfAddr = substr($fullPdu, $index, 2);
        $index += 2;

        // Step 6: Sender number semi-octets (length in octets = ceil(senderLen/2))
        $senderNumOctets = ceil($senderLenSemiOctets / 2) * 2;
        $senderSemiOctets = substr($fullPdu, $index, $senderNumOctets);
        $index += $senderNumOctets;

        // Decode sender number semi-octets to string
        $senderNumber = $this->semiOctetToString($senderSemiOctets);
        // Remove trailing 'F' padding if exists
        $senderNumber = rtrim($senderNumber, 'F');

        // If sender is international number (type starts with 91), prepend +
        if (strpos($senderTypeOfAddr, '91') === 0) {
            $senderNumber = '+' . $senderNumber;
        }

        // Step 7: Protocol Identifier (1 octet) - skip
        $index += 2;

        // Step 8: Data Coding Scheme (1 octet)
        $dcs = hexdec(substr($fullPdu, $index, 2));
        $index += 2;

        // Step 9: Timestamp (7 octets, 14 hex digits)
        $timestampSemiOctets = substr($fullPdu, $index, 14);
        $index += 14;

        // Decode timestamp from semi-octets to datetime string
        $timestamp = $this->decodeTimestamp($timestampSemiOctets);

        // Step 10: User Data Length (1 octet)
        $udl = hexdec(substr($fullPdu, $index, 2));
        $index += 2;

        // Step 11: User Data (variable length)
        $userDataHex = substr($fullPdu, $index);

        // Detect encoding
        $encoding = null;
        if (($dcs & 0x0C) == 0x08) {
            $encoding = 'ucs2';
        } elseif (($dcs & 0x0C) == 0x04) {
            $encoding = '8bit';
        } else {
            $encoding = '7bit';
        }

        // Adjust user data length in hex chars depending on encoding
        if ($encoding == '7bit') {
            $numHexChars = ceil($udl * 7 / 8) * 2;
            $userDataHex = substr($userDataHex, 0, $numHexChars);
        } else {
            $numHexChars = $udl * 2;
            $userDataHex = substr($userDataHex, 0, $numHexChars);
        }

        // Decode message text using your existing methods
        switch ($encoding) {
            case 'ucs2':
                $message = $this->decodeUCS2($userDataHex);
                break;
            case '8bit':
                $message = $this->decode8Bit($userDataHex);
                break;
            case '7bit':
            default:
                $message = $this->decodeGSM7Bit($userDataHex);
                break;
        }

        return [
            'sender' => $senderNumber,
            'timestamp' => $timestamp,
            'dcs' => $dcs,
            'encoding' => $encoding,
            'message' => $message,
        ];
    }

    /**
     * Decodes semi-octet timestamp to human-readable format
     *
     * @param string $semiOctets 14 hex chars representing 7 semi-octets timestamp
     * @return string datetime in Y-m-d H:i:s format
     */
    public function decodeTimestamp(string $semiOctets): string
    {
        $digits = str_split($semiOctets, 2);

        $year = $this->swapNibbles($digits[0]);
        $month = $this->swapNibbles($digits[1]);
        $day = $this->swapNibbles($digits[2]);
        $hour = $this->swapNibbles($digits[3]);
        $minute = $this->swapNibbles($digits[4]);
        $second = $this->swapNibbles($digits[5]);

        $yearFull = 2000 + intval($year);

        return sprintf('%04d-%02d-%02d %02d:%02d:%02d', $yearFull, $month, $day, $hour, $minute, $second);
    }

    /**
     * Swap the nibbles in a hex byte (e.g. '52' -> '25')
     *
     * @param string $byte 2 hex chars
     * @return string
     */
    protected function swapNibbles(string $byte): string
    {
        return $byte[1] . $byte[0];
    }

    // Your existing methods below (decodeGSM7Bit, decodeUCS2, decode8Bit, etc.) unchanged:

    public function decodeGSM7Bit($message, $skip = 0)
    {
        $length = strlen($message);
        if (($length - $skip) % 2) {
            throw new \InvalidArgumentException("Given text is not well formatted. \"" . substr($message, $skip) . "\"");
        }

        $septets = floor($length / 2 * 8 / 7);
        $buffer = $this->decodeMessage7Bit($skip, $message, $septets);
        $length = strlen($buffer);
        $padding = chr(0x0D);
        if (($septets % 8 == 0 && $length > 0 && $buffer[$length - 1] == $padding) ||
            ($septets % 8 == 1 && $length > 1 && $buffer[$length - 1] == $padding && $buffer[$length - 2] == $padding)) {
            $buffer = substr($buffer, 0, $length - 1);
        }

        return $buffer;
    }

    public function decodeUCS2($encoded)
    {
        $length = strlen($encoded);
        if ($length % 2) {
            throw new \InvalidArgumentException("Given text is not well formatted.");
        }

        return $this->decodeMessage16Bit($encoded);
    }

    public function decode8Bit($encoded)
    {
        $length = strlen($encoded);
        if ($length % 2) {
            throw new \InvalidArgumentException("Given text is not well formatted.");
        }
        return $this->decodeMessage8Bit($encoded, $length);
    }

    protected function decodeMessage7Bit($skip, $encoded, $length)
    {
        $message = "";

        $octets = array();
        $rest = array();
        $septets = array();

        $s = 1;
        $count = 0;
        $matchcount = 0;
        $escaped = false;
        $chars = 0;

        $byteString = "";
        for ($i = 0; $i < strlen($encoded); $i += 2) {
            $hex = substr($encoded, $i, 2);
            $byteString .= sprintf('%08b', hexdec($hex));
        }

        for ($i = 0; $i < strlen($byteString); $i += 8) {
            $octets[$count] = substr($byteString, $i, 8);
            $rest[$count] = substr($octets[$count], 0, ($s % 8));
            $septets[$count] = substr($octets[$count], ($s % 8));

            $s++;
            $count++;

            if ($s == 8) {
                $s = 1;
            }
        }

        for ($i = 0; $i < count($rest); $i++) {
            if (($i % 7) == 0) {
                if ($i != 0) {
                    $chars++;
                    $chrVal = bindec($rest[$i - 1]);

                    if ($escaped) {
                        $message .= $this->getSevenBitExtendedChar($chrVal);
                        $escaped = false;
                    } else if ($chrVal == 27 && $chars > $skip) {
                        $escaped = true;
                    } else if ($chars > $skip) {
                        $message .= static::$sevenbitdefault[$chrVal];
                    }

                    $matchcount++;
                }

                $chars++;
                $chrVal = bindec($septets[$i]);
                if ($escaped) {
                    $message .= $this->getSevenBitExtendedChar($chrVal);
                    $escaped = false;
                } else if ($chrVal == 27 && $chars > $skip) {
                    $escaped = true;
                } else if ($chars > $skip) {
                    $message .= static::$sevenbitdefault[$chrVal];
                }

                $matchcount++;
            } else {
                $chars++;

                $chrVal = bindec($septets[$i] . $rest[$i - 1]);

                if ($escaped) {
                    $message .= $this->getSevenBitExtendedChar($chrVal);
                    $escaped = false;
                } else if ($chrVal == 27 && $chars > $skip) {
                    $escaped = true;
                } else if ($chars > $skip) {
                    $message .= static::$sevenbitdefault[$chrVal];
                }

                $matchcount++;
            }
        }

        if ($matchcount != $length) {
            $chars++;
            $chrVal = bindec($rest[$i - 1]);
            if (!$escaped) {
                if ($chars > $skip) {
                    $message .= static::$sevenbitdefault[$chrVal];
                }
            } else if ($chars > $skip) {
                $message .= $this->getSevenBitExtendedChar($chrVal);
            }
        }

        return $message;
    }

    protected function decodeMessage16Bit($encoded, $skip = 0)
    {
        if ($skip > 0) {
            $encoded = mb_substr($encoded, $skip);
        }
        return mb_convert_encoding(pack('H*', $encoded), 'UTF-8', 'UCS-2');
    }

    protected function decodeMessage8Bit($encoded, $length)
    {
        $message = "";
        for ($i = 0; $i < $length; $i += 2) {
            $hex = substr($encoded, $i, 2);
            $message .= chr(hexdec($hex));
        }

        return $message;
    }

    public function getSevenBitExtendedChar($code)
    {
        $key = array_search($code, static::$sevenbitextended);

        if ($key !== false) {
            return $key;
        }

        return "\u2639"; // sad face unicode fallback
    }

    public function getSevenBitExtendedCode($char)
    {
        if (array_key_exists($char, static::$sevenbitextended)) {
            return static::$sevenbitextended[$char];
        }

        return 0;
    }

    public function getSevenBitCode($char)
    {
        return (int)array_search($char, static::$sevenbitdefault);
    }

    public function encodeGSM7Bit($message)
    {
        if (!static::isGSM7Bit($message)) {
            throw new \InvalidArgumentException("Given string is not 7-Bit GSM compatible!");
        }

        $encoded = "";
        $padding = chr(0x0D);
        $tmp = $message;
        $message = "";
        $firstOctet = "";
        $secondOctet = "";

        for ($i = 0; $i < mb_strlen($tmp); $i++) {
            $char = mb_substr($tmp, $i, 1);
            if ($this->getSevenBitExtendedCode($char)) {
                $message .= chr(0x1B);
            }

            $message .= $char;
        }

        $length = mb_strlen($message);

        if (($length % 8) == 7 || (($length % 8) == 0 && $length > 0 && mb_substr($message, -1) == $padding)) {
            $message .= $padding;
        }

        for ($i = 0; $i <= mb_strlen($message); $i++) {
            if ($i == mb_strlen($message)) {
                if ($secondOctet != "") {
                    $encoded .= sprintf('%02s', dechex(bindec($secondOctet)));
                }
                break;
            }

            $char = mb_substr($message, $i, 1);

            if ($char == chr(0x1B)) {
                $current = sprintf('%07b', 0x1B);
            } else {
                $tmp = $this->getSevenBitExtendedCode($char);
                if ($tmp == 0) {
                    $tmp = $this->getSevenBitCode($char);
                }

                $current = sprintf('%07b', $tmp);
            }

            $currentOctet = "";

            if ($i != 0 && ($i % 8) != 0) {
                $firstOctet = mb_substr($current, 7 - ($i % 8));
                $currentOctet = $firstOctet . $secondOctet;

                $encoded .= sprintf('%02s', dechex(bindec($currentOctet)));
                $secondOctet = mb_substr($current, 0, 7 - ($i % 8));
            } else {
                $secondOctet = mb_substr($current, 0, 7 - ($i % 8));
            }
        }

        return strtoupper($encoded);
    }

    public function encodeUCS2($message)
    {
        return strtoupper(bin2hex(mb_convert_encoding($message, 'UCS-2', 'UTF-8')));
    }

    public function encodeGSM8Bit($message)
    {
        throw new \RuntimeException('Not supported yet.');
    }

    public static function isGSM7Bit($message)
    {
        $gsm7bitChars = join(static::$sevenbitdefault) . join(array_keys(static::$sevenbitextended));
        for ($i = 0; $i < mb_strlen($message); $i++) {
            $char = mb_substr($message, $i, 1);
            if (mb_strpos($gsm7bitChars, $char) === false && $char != "\\") {
                return false;
            }
        }
        return true;
    }

    public static function getValidEncodings()
    {
        $validEncodings = array();
        $refClass = new \ReflectionClass(__CLASS__);
        foreach ($refClass->getConstants() as $name => $value) {
            if (substr($name, 0, 9) == 'ENCODING_') {
                array_push($validEncodings, $value);
            }
        }
        return $validEncodings;
    }

    public function semiOctetToString($input)
    {
        $out = "";
        for ($i = 0; $i < strlen($input); $i += 2) {
            $temp = substr($input, $i, 2);
            $out .= $this->phoneNumberMap($temp[1]) . $this->phoneNumberMap($temp[0]);
        }

        return $out;
    }

    protected function phoneNumberMap($char)
    {
        if (is_numeric($char) && $char >= 0 && $char <= 9) {
            return $char;
        }

        switch (strtoupper($char)) {
            case '*':
                return 'A';
            case '#':
                return 'B';
            case 'A':
                return 'C';
            case 'B':
                return 'D';
            case 'C':
                return 'E';
            case '+':
                return '+';
            default:
                return 'F';
        }
    }
}
