<?php

namespace App\Http\Controllers;

use Carbon\Carbon;
use App\Models\ModemInfo;
use App\Models\DevicePort;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;

class PortController extends Controller
{
    // Helper function to convert 'NULL' strings to null
    private function cleanNullString($value)
    {
        if (is_string($value) && strtolower($value) === 'null') {
            return null;
        }
        return $value;
    }


    public function index() {
    $data = DevicePort::all();  // Fetch all records
    return view('Device', ['data' => $data]);  // Pass data to the view
}




// public function device_port(Request $request)
// {
//     $data = $request->all();

//     Log::info('Raw input data:', $data);

//     return response()->json($data);
// }

public function saveModemData(Request $request)
{
    Log::info('Received modem data: ' . $request->getContent());

    $input = $request->all();

    // Basic validation for required fields
    $validator = Validator::make($input, [
        'apikey' => 'required|string',
        'equipment' => 'required|string',
        'usbline' => 'sometimes|array',
        'portdata' => 'sometimes|array',
    ]);

    if ($validator->fails()) {
        Log::warning('Validation failed: ' . json_encode($validator->errors()->toArray()));
        return response()->json([
            'status' => 'error',
            'message' => 'Invalid data',
            'errors' => $validator->errors(),
        ], 422);
    }

    try {
        // Organize Modem Info data
        $modemInfoData = null;
        if (isset($input['usbline']) && is_array($input['usbline'])) {
            $modemInfoData = [
                'api_key' => $input['apikey'],
                'equipment_uuid' => $input['equipment'],
                'number_of_ports' => count($input['usbline']),
                'usbline' => $input['usbline'],
            ];
        }

        // Organize Port data
        $organizedPorts = [];
        if (isset($input['portdata']) && is_array($input['portdata'])) {
            foreach ($input['portdata'] as $portName => $portData) {
                // Normalize empty strings to null for critical fields
                $modemModel = isset($portData['AT+CGMM']) && $portData['AT+CGMM'] !== '' ? $portData['AT+CGMM'] : null;
                $simPinRaw = $portData['AT+CPIN'] ?? null;
                $simPin = ($simPinRaw == 1) ? 1 : 0;

                $organizedPorts[$portName] = [
                    'imei' => $portData['imei'] ?? null,
                    'modem_model' => $modemModel,
                    'sim_iccid' => $portData['AT+QCCID'] ?? null,
                    'functionality' => $portData['AT+CFUN'] ?? null,
                    'sim_pin_status' => $simPin,
                    'subscriber_imsi' => $portData['AT+CIMI'] ?? null,
                    'network_operator' => $portData['AT+COPS'] ?? null,
                    'signal_quality' => $portData['AT+CSQ'] ?? null,
                    'timer_seconds' => $portData['timer'] ?? null,
                    'last_update' => (isset($portData['lastupdate']) && $portData['lastupdate'] > 0)
                        ? \Carbon\Carbon::createFromTimestamp($portData['lastupdate'])
                        : null,
                    'operator_errors' => $portData['COPSERROR'] ?? 0,
                ];
            }
        }

        // Log organized data for debugging
        Log::info('Organized Modem Info:', $modemInfoData ?: []);
        Log::info('Organized Port Data:', $organizedPorts);

        // Upsert ModemInfo
        if ($modemInfoData) {
            $modem = ModemInfo::updateOrCreate(
                ['equipment_uuid' => $modemInfoData['equipment_uuid']],
                [
                    'api_key' => $modemInfoData['api_key'],
                    'number_of_ports' => $modemInfoData['number_of_ports'],
                    'updated_at' => now(),
                ]
            );
            Log::info("ModemInfo upserted: ID {$modem->id}");
        } else {
            // Fetch existing modem if usbline missing
            $modem = ModemInfo::where('equipment_uuid', $input['equipment'])->first();
            if (!$modem) {
                Log::warning('ModemInfo not found for equipment: ' . $input['equipment']);
                return response()->json([
                    'status' => 'error',
                    'message' => 'Modem info not found and usbline missing.',
                ], 400);
            }
        }

        // Upsert DevicePorts
        if (!empty($organizedPorts)) {
            foreach ($organizedPorts as $portName => $portData) {
                DevicePort::updateOrCreate(
                    [
                        'modem_info_id' => $modem->id,
                        'port_name' => $portName,
                    ],
                    array_merge($portData, ['updated_at' => now()])
                );
            }
            Log::info('Device ports updated for modem ID: ' . $modem->id);
        } else {
            Log::warning('No port data to update');
        }

        return response()->json([
            'status' => 'success',
            'message' => 'Modem data saved successfully.',
            'modem_id' => $modem->id,
        ]);
    } catch (\Exception $e) {
        Log::error('Failed to save modem data: ' . $e->getMessage());

        return response()->json([
            'status' => 'error',
            'message' => 'Failed to save modem data.',
            'error' => $e->getMessage(),
        ], 500);
    }
}



}

